/*
  OKI Menu Control Program
  Copyright (C) 1994 by Network Wizards
  by Mark K. Lottor
*/

#define FALSE		0
#define TRUE		1

/* pick version of program */
#define PC		TRUE
#define UNIX		FALSE

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <string.h>
#include <time.h>

#if PC
#include <dos.h>
#include <conio.h>
#include <bios.h>
#endif

#if UNIX
#include <sys/types.h>
#include <fcntl.h>
#include <sgtty.h>
#endif


#define VERSION	    "1.3(940112)"   /* major.minor(last-edit-date) */

typedef unsigned char bool;
typedef unsigned char byte;
#if PC
typedef unsigned int  word;		/* 16 bits */
#endif
#if UNIX
typedef unsigned short word;		/* 16 bits */
#endif

#if PC
#include "ctlib.h"
#endif
#include "menu.dat"
#include "memories.h"

/* key definitions */
#define CNB		0x02
#define CNF		0x06
#define CNN		0x0e
#define CNP		0x10
#define BS		0x08
#define LF		0x0a
#define CR		0x0d
#define ESC		0x1B
#define TAB		0x09
#define INS		0x01
#define DEL		0x7f
#define PUP		0x0b
#define PDN		0x0a

#define LEGAL_ALPHAS	"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789*#&.?'+()/-=`~!@$%^_{}[];:,<>|\\\" "
#define LEGAL_NUMS	"0123456789*#P"
char MEM_DEFS[]   = ".1234567890*#..P";

/* EEPROM locations (bytes) */
#define EE_NAMBLOCK_9	0xa02b		/* nam block (by 0x40s) */
#define EE_NAMINC_9     0x40
#define EE_NAMBLOCK_11	0xff80		/* nam block (by 0x01s) */
#define EE_NAMINC_11    1
#define EE_NUMBITS1_9   0xbeec		/* tel num memory in use bitmap */
#define EE_NUMBITS2_9   0xbf05		/* duplicate of above */

word menu_eenamblock;
byte menu_eenaminc;

#define BUFLEN		128
  
#define INVERSE_VIDEO	0x70
#define NORMAL_VIDEO	0x07

#if PC
#define HORIZ_LINE      '\315'
#define VERT_LINE       '\263'
#define CROSS           '\330'
#endif
#if UNIX
#define HORIZ_LINE      '-'
#define VERT_LINE       '|'
#define CROSS           '+'
#endif

/* function to position cursor on screen */
#if PC
#define cursor(x,y)	gotoxy(y,x)
#endif
#if UNIX
#define cprintf		printf
#define putch           putchar
#define getch           getchar
#define getche          getchar
#define stdprn          stdout
#endif

byte cursor_row, cursor_col;
char status_string[40];
char args[BUFLEN];
char last_file[16];

char esn[16];				/* esn in ascii hex */
int esn_man;
long esn_ser;
char romver[16];			/* rom version in ascii hex */
char dlrpwd[16];			/* dealer password */
byte custmsg[16];			/* custom poweron msg */
byte xnamblock[80];			/* nam info block */
/* xnamblock data turned into printable ascii */
struct xnam_data {
  char number[16];
  char sid[8];
  char sys;
  char ipch[8];
  char olc[8];
  char gim[8];
} xnam_info[5];


char buf[BUFLEN];
char ubuf[BUFLEN];

struct stack_data {
  struct menu_data *menu;
  int item;
} stack[16];
int stack_ptr = 0;

#define MAX_MEMORIES	200
struct memory_data {
  bool got;
  bool mod;
  char name[11];
  char number[33];
} memories[MAX_MEMORIES];
#define MEMORIES_SIZE sizeof(struct memory_data)
int max_memories;
int max_namelen;

bool edited = FALSE;
int  edit_pos = 0;
char edit_string[64];

/* forward declarations */
void change_nam();

main(argc,argv)
  int argc;
  char *argv[];
{
  int i, comport;
  byte cmd, data;
  word msg, key;
#if UNIX
  struct sgttyb ttyargs;
#endif

  MEM_DEFS[0] = '\0';
  *last_file = '\0';
  draw_screen();			/* display initial screen */
#if PC
  if (argc > 1)
  {
    if (*argv[1] == '1')
      ct_lib_init(900,0x3f8,4);
    else if (*argv[1] == '2')
      ct_lib_init(900,0x2f8,3);
    else if (*argv[1] == '3')
      ct_lib_init(900,0x3e8,5);
    else
    {
      cputs("Type 'MENU 2' to use COM2");
      exit(0);
    }
  }
  else
    ct_lib_init(900,0x3f8,4);
#endif
#if UNIX
  tty_init(argv[1]);			/* open /dev/ttyxx file */

  ttyargs.sg_ispeed = B9600;
  ttyargs.sg_ospeed = B9600;
  ttyargs.sg_erase = 0;
  ttyargs.sg_kill = 0;
  ttyargs.sg_flags = RAW;
  if ((ioctl(0, TIOCSETP, (char *) &ttyargs)) == -1)
  {
    fprintf(stderr,"?Can't do ioctl TIOCSETP on tty.\n");
    exit(1);
  }
#endif

  /* power up oki and tell it what we are */
  if (!ct_on(MODE_TEST))
  {
    cursor(25,1);
    cprintf("?No response from OKI.\r\n");
    cprintf("  Make sure the phone is turned OFF and\r\n");
    cprintf("  all cables are properly attached.\r\n\n");
    cleanup();
    exit(1);
  }
  ct_power(PWR_TX,FALSE);

  display_menu_title_and_help("Initializing...\\Please wait");
  
  if (ct_model == 900)
  {
    menu_eenaminc   = EE_NAMINC_9;
    menu_eenamblock = EE_NAMBLOCK_9;
    max_memories    = 200;
    max_namelen     = 8;
  }
  else if (ct_model >= 1150)
  {
    menu_eenaminc   = EE_NAMINC_11;
    menu_eenamblock = EE_NAMBLOCK_11;
    max_memories    = 100;
    max_namelen     = 10;
  }
  else
  {
    status("?Model of phone not recognized.");
    cleanup();
    exit(1);
  }

  get_esn();
  strcpy(romver,ct_get_romver());
  strcpy(dlrpwd,ct_get_prog_pwd());
  strcpy(custmsg,ct_get_cust_msg());
  get_xnamblock();
  gen_naminfo();
  
  current_menu = main_menu;
  display_menu();
  current_item = 1;
  display_item(current_item,INVERSE_VIDEO);
  for (i = 0; i < MAX_MEMORIES; i++)
  {
    memories[i].got = FALSE;
    memories[i].mod = FALSE;
    *memories[i].name = '\0';
    *memories[i].number = '\0';
  }
  /* main command loop */ 
  for (;;)
  {
#if PC
    if (bioskey(1) != 0)		/* if keypress waiting */
    {
      key = bioskey(0);			/* read keypress codes */
      cmd = key & 0x00ff;		/* get ascii code if any */
      data = (key & 0xff00) >> 8;	/* key function code if any */
#endif
#if UNIX
    if (TRUE)
    {
      read(0,&cmd,1);		        /* get keypress */
      cmd = cmd & 0x7f;
#endif
      if ((cmd == ESC) && (current_menu == main_menu))
	break;
      if (isupper(cmd))
	cmd = tolower(cmd);
      if (cmd == 0)			/* if no ascii code */
      {
	switch (data)			/* convert function code to ascii */
	{
	  /* arrow keys or emacs style cursor movement */
	  case 0x48: cmd = CNP; break;
	  case 0x4b: cmd = CNB; break;
	  case 0x4d: cmd = CNF; break;
	  case 0x50: cmd = CNN; break;
	  deault: break;
	}
      }
      switch (cmd)
      {
	case CNP: display_item(current_item,NORMAL_VIDEO);
		  if (current_item == 1) current_item = current_last_item;
		    else current_item--;
	          display_item(current_item,INVERSE_VIDEO);
		  break;
	case CNN: display_item(current_item,NORMAL_VIDEO);
		  if (current_item == current_last_item) current_item = 1;
		    else current_item++;
	          display_item(current_item,INVERSE_VIDEO);
		  break;
        case CNF:
        case CR:
        case LF:
	          if (current_menu[current_item].menu != NULL)
		  {
		    push_menu(current_menu,current_item);
		    current_menu = current_menu[current_item].menu;
		    current_item = 1;
		    display_menu();
		    display_item(current_item,INVERSE_VIDEO);
		    break;
		  }
	          if (current_menu[current_item].func != NULL)
		  {
		    push_menu(current_menu,current_item);
		    clear_menu_area();
		    (*current_menu[current_item].func)(current_item);
		    pop_menu(&current_menu,&current_item);
		    display_menu();
		    display_item(current_item,INVERSE_VIDEO);
		  }
		  break;
        case CNB: pop_menu(&current_menu,&current_item);
		  display_menu();
	          display_item(current_item,INVERSE_VIDEO);
		  break;
	default:  break;
      }
    }
  }
  cleanup();
  exit(0);
}

cleanup()
{
#if PC
  ct_off();
  ct_lib_done();
  cursor(24,1);  cputs("                                \r");
#endif
#if UNIX
  tty_done();
  cursor(22,1);
#endif
}

push_menu(menu,item)
  struct menu_data *menu;
  int item;
{
  stack[stack_ptr].menu = menu;
  stack[stack_ptr].item = item;
  stack_ptr++;
}

pop_menu(menu,item)
  struct menu_data **menu;
  int *item;
{
  if (stack_ptr <= 0) return;
  stack_ptr--;
  *menu = stack[stack_ptr].menu;
  *item = stack[stack_ptr].item;
}

display_item(item,attrib)
  int item;
  byte attrib;
{
  char *p;
  
  textattr(attrib);
  cursor(5+item,1);
  p = current_menu[item].text;
  while (*p != '\0')
  {
    if (*p == '*') putch('\032');
      else putch(*p);
    p++;
  }
  textattr(NORMAL_VIDEO);
#if UNIX
  fflush(stdout);
#endif
}

clear_menu_area()
{
  int i;
  for (i = 6; i < 24; i++)
  {
    cursor(i,1);  cputs("                                               ");
  }
#if UNIX
  fflush(stdout);
#endif
}

display_menu()
{
  int i, c;
  char *p, *q;
  
  display_menu_title_and_help(current_menu[0].text);
  i = 1;
  while (current_menu[i].text != NULL)
  {
    cursor(5+i,1);
    p = current_menu[i].text;
    c = 1;
    while (*p != '\0')
    {
      if (*p == '*') putch('\032');
        else putch(*p);
      p++;
      c++;
    }
    while (c < 48) {putch(' '); c++;}
    i++;
  }
  current_last_item = i - 1;
  while (i < 21)
  {
    cursor(5+i,1);
    cputs("                                             ");
    i++;
  }
#if UNIX
  fflush(stdout);
#endif
}

display_menu_title_and_help(msg)
  char *msg;
{
  int i, c;
  char *q;
  
  cursor(1,1);
  q = msg;
  c = 1;
  while (*q != '\\') { putch(*q++); c++; }
  while (c < 48) {putch(' '); c++;}
  i = 6;
  while (*q++ != '\0')
  {
    cursor(i,49);
    c = 49;
    while ((*q != '\0') && (*q != '\\'))
      {putch(*q++); c++;}
    while (c < 80) {putch(' '); c++;}
    i++;
  }
  while (i < 26)
  {
    cursor(i,49);
    cputs("                               ");
    i++;
  }
#if UNIX
  fflush(stdout);
#endif
}

void read_memories()
{
  int i, mem, n1, n2;
  char mems[4], *p;
  byte cmd, data;
  word addr;
  
  display_menu_title_and_help("Reading Memories from Phone\\\
Hit any key to interrupt.\\\
\\\
After memories have been read\\\
select Edit Memories function\\\
to change them.\\\
\\\
When done editing, select\\\
Write Memories function\\\
to save them in phone.");
  cursor(6,1); cputs("Reading memory ");
  ct_rx_flush();
  fflush(stdin);
  while (kbhit()) getch();
  for (mem = 0; mem < max_memories; mem++)
  {
    if (kbhit()) break;
    if (memories[mem].got) continue;
    
    /* read number */
    if (ct_model == 900)
      addr = mem_number_locs[mem];	/* get address in eeprom space */
    else
      addr = 0xf0c0 + (mem * 32);
    for (i = 0; i < 16; i++)
    {
      ct_oki_send_data(25);
      if (ct_model >= 1150)
	ct_oki_send_data(0);
      ct_oki_send_data((addr >> 8) & 0xff);
      ct_oki_send_data( addr       & 0xff);
      incaddr(&addr);
      if (!ct_oki_get_msg(&cmd,&data,1000)) break;
      n1 = (data >> 4);
      n2 = data & 0x0f;
      memories[mem].number[i*2] =   MEM_DEFS[n1];
      memories[mem].number[i*2+1] = MEM_DEFS[n2];
      if ((n1 == 0) || (n2 == 0))
	break;
    }
    memories[mem].number[33] = '\0';

    /* read name */
    if (ct_model == 900)
      addr = mem_name_locs[mem];	/* get address in eeprom space */
    else
      addr = 0xf0d0 + (mem * 32);
  if ((i != 0) || (n1 != 0))   /* if there was a number, then get the name */
    for (i = 0; i < max_namelen;)
    {
      ct_oki_send_data(25);
      if (ct_model >= 1150)
	ct_oki_send_data(0);
      ct_oki_send_data((addr >> 8) & 0xff);
      ct_oki_send_data( addr       & 0xff);
      incaddr(&addr);
      ct_oki_send_data(25);
      if (ct_model >= 1150)
	ct_oki_send_data(0);
      ct_oki_send_data((addr >> 8) & 0xff);
      ct_oki_send_data( addr       & 0xff);
      incaddr(&addr);
      if (!ct_oki_get_msg(&cmd,&data,1000)) break;
      memories[mem].name[i] = data;
      if (data == 0)
      {
	ct_oki_get_msg(&cmd,&data,1000);
	break;
      }
      i++;
      if (!ct_oki_get_msg(&cmd,&data,1000)) break;
      memories[mem].name[i] = data;
      if (data == 0) break;
      i++;
    }
    memories[mem].name[max_namelen] = '\0';
    memories[mem].got = TRUE;

    sprintf(mems,"%03d",mem);
    cursor(6,16); cprintf("%s %s        ",mems,memories[mem].name);
#if UNIX
    fflush(stdout);
#endif
  }
  cursor(2,1); cputs("                            ");
}

/* Weird! */
incaddr(addr)
  word *addr;
{
  for (;;)
  {
    (*addr)++;
    if (ct_model >= 1150)
      break;
    if ((*addr & 0x003e) != 0x002a)
      break;
  }
}

void write_memories()
{
  int i, mem, pos, mpos, pushes, end;
  char mems[4], *p, *q, digit, last_digit;
  byte cmd, data, c, sum, bits;
  bool lowerp;
  word addr, bitaddr;
  
  display_menu_title_and_help("Write Memories to Phone\\\
Hit ESCAPE to abort.");
  cursor(6,1); cputs("Writing memory ");
  ct_rx_flush();
  for (mem = 0; mem < max_memories; mem++)
  {
    if (kbhit())
      if (getch() == ESC) break;
    sprintf(mems,"%03d",mem);
    cursor(6,16); cputs(mems);
    if (!memories[mem].mod) continue;
    ct_rx_flush();

    sum = 0;
    /* write number */
    if (ct_model == 900)
      addr = mem_number_locs[mem];	/* get address in eeprom space */
    else
      addr = 0xf0c0 + (mem * 32);
    for (i = 0; i < 16; i++)
    {
      ct_oki_send_data(54);
      if (ct_model >= 1150)
	ct_oki_send_data(0);
      ct_oki_send_data((addr >> 8) & 0xff);
      ct_oki_send_data( addr       & 0xff);
      incaddr(&addr);
      c = memories[mem].number[i*2];
      if (c == ' ') c = 0;
       else if (c == '0') c = 0x0a;
        else if (c == '*') c = 0x0b;
         else if (c == '#') c = 0x0c;
          else if (c == 'P') c = 0x0f;
      data = (c & 0x0f) << 4;
      if (c > 0) sum++;
      c = memories[mem].number[i*2+1];
      if (c == ' ') c = 0;
       else if (c == '0') c = 0x0a;
        else if (c == '*') c = 0x0b;
         else if (c == '#') c = 0x0c;
          else if (c == 'P') c = 0x0f;
      data = data + (c & 0x0f);
      if (c > 0) sum++;
      ct_oki_send_data(data);
      sum = sum + data;
      ct_get_rss();			/* delay till write occurs */
    }

    /* write name */
    if (ct_model == 900)
      addr = mem_name_locs[mem];	/* get address in eeprom space */
    else
      addr = 0xf0d0 + (mem * 32);
    for (i = 0; i < max_namelen; i++)
    {
      ct_oki_send_data(54);
      if (ct_model >= 1150)
	ct_oki_send_data(0);
      ct_oki_send_data((addr >> 8) & 0xff);
      ct_oki_send_data( addr       & 0xff);
      incaddr(&addr);
      c = memories[mem].name[i];
      if (c == 0) c = ' ';
      ct_oki_send_data(c);
      sum = sum + c;
      ct_get_rss();			/* delay till write occurs */
    }

    /* write check digit right after name */
    sum = (~sum) + 1;
    ct_oki_send_data(54);
      if (ct_model >= 1150)
	ct_oki_send_data(0);
    ct_oki_send_data((addr >> 8) & 0xff);
    ct_oki_send_data( addr       & 0xff);
    ct_oki_send_data(sum & 0xff);

    if (ct_model == 900)
    {
      /* update bitmaps */
      /* first read bits */
      bitaddr = EE_NUMBITS1_9 + (mem/8);
      ct_oki_send_data(25);			
      ct_oki_send_data((bitaddr >> 8) & 0xff);
      ct_oki_send_data( bitaddr       & 0xff);
      ct_oki_get_msg(&cmd,&data,2000);
      data = data | (1 << (mem % 8));	/* set new bit */

      /* update bit maps */
      ct_oki_send_data(54);
      ct_oki_send_data((bitaddr >> 8) & 0xff);
      ct_oki_send_data( bitaddr       & 0xff);
      ct_oki_send_data(data);
      ct_get_rss();			/* delay till write occurs */
      bitaddr = EE_NUMBITS2_9 + (mem/8);
      ct_oki_send_data(54);
      ct_oki_send_data((bitaddr >> 8) & 0xff);
      ct_oki_send_data( bitaddr       & 0xff);
      ct_oki_send_data(data);
    }
    
    memories[mem].mod = FALSE;
    ct_get_rss();			/* delay till write occurs */
  }
  cursor(6,1); cputs("Memories in phone have been updated.");
  cursor(7,1); cputs("Hit any key to continue.");
  getch();
  cursor(6,1); cputs("                                    ");
  cursor(7,1); cputs("                        ");
  ct_rx_flush();
}

get_esn()
{
  byte cmd, data;
  word a, b, c, d;
  unsigned long n;

  if (!ct_get_esn(&n))
    n = 0;
  sprintf(esn,"%08lX",n);
  esn_man = n >> 24;
  esn_ser = n & 0xffffff;
}

/* read 67 byte checksummed nam info block */
get_xnamblock()
{
  int i, sum;
  byte cmd, data;
  word addr;
  
  addr = menu_eenamblock;
  for (i = 0; i < 67; i++)
  {
    ct_oki_send_data(25);
    if (ct_model >= 1150)
      ct_oki_send_data(0);
    ct_oki_send_data((addr >> 8) & 0xff);
    ct_oki_send_data( addr       & 0xff);
    addr = addr + menu_eenaminc;
    ct_oki_send_data(25);
    if (ct_model >= 1150)
      ct_oki_send_data(0);
    ct_oki_send_data((addr >> 8) & 0xff);
    ct_oki_send_data( addr       & 0xff);
    addr = addr + menu_eenaminc;
    if (!ct_oki_get_msg(&cmd,&data,1000)) return;
    xnamblock[i] = data;
    i++;
    if (!ct_oki_get_msg(&cmd,&data,1000)) return;
    xnamblock[i] = data;
  }
  /* test for correct checksum */
  sum = 0;
  for (i = 0; i < 67; i++)
  {
    sum = sum + xnamblock[i];
  }
  sum = sum & 0xff;
  if (sum != 0)
  {
    printf("NAM checksum incorrect!!!");
    exit(1);
  }
}

gen_naminfo()
{
  int i, ac, pre, mid, end, sid;
  
  for (i = 0; i < 5; i++)
  {
    ac  = (xnamblock[i*11+2] << 4) + (xnamblock[i*11+3] >> 4);
    pre = ((xnamblock[i*11+3] & 0x0f) << 6) + (xnamblock[i*11+4] >> 2);
    mid = ((xnamblock[i*11+4] & 0x03) << 2) + (xnamblock[i*11+5] >> 6);
    if (mid == 10) mid = 0;
    end = ((xnamblock[i*11+5] & 0x3f) << 4) + (xnamblock[i*11+6] >> 4);
    sprintf(xnam_info[i].number,"(%03d) %03d-%d%03d",
                ct_minunpack(ac),ct_minunpack(pre),mid,ct_minunpack(end));
    sid = (xnamblock[i*11] << 8) + xnamblock[i*11+1];
    sprintf(xnam_info[i].sid,"%05d",sid);
    if (sid & 0x0001) 
      xnam_info[i].sys = 'A';
    else
      xnam_info[i].sys = 'B';
    sprintf(xnam_info[i].ipch,"%04d",
                             (xnamblock[i*11+7] << 8) + xnamblock[i*11+8]);
    sprintf(xnam_info[i].olc,"%02d",xnamblock[i*11+9]);
    sprintf(xnam_info[i].gim,"%02d",xnamblock[i*11+10]);

    sprintf(nam_menu[i+1].text,"NAM %d:  %s",i+1,xnam_info[i].number);
  }
}

void change_nam(nam)
  int nam;
{
  char c, s[8];
  char *p;
  int i, n;
  
  display_menu_title_and_help("Change NAM Information\\\
Enter the new NAM information\\\
at each prompt.\\\
\\\
Hit RETURN to accept the\\\
default answer which is\\\
displayed in brackets.");
 for (;;)
 {
  cursor(6,1);  cprintf("Current setting for NAM %d:",nam);
  cursor(7,1);  cprintf("  Telephone Number:        %s",xnam_info[nam-1].number);
  cursor(8,1);  cprintf("  System ID:               %s",xnam_info[nam-1].sid);
  cursor(9,1);  cprintf("  Initial Paging Channel:  %s",xnam_info[nam-1].ipch);
  cursor(10,1); cprintf("  Access Overload Class:   %s",xnam_info[nam-1].olc);
  cursor(11,1); cprintf("  Group ID Mark:           %s",xnam_info[nam-1].gim);
  cursor(13,1); cprintf("Change (Y/N)? ");
  c = getch();
  if ((c == 'n') || (c == 'N')) break;
  if ((c == 'y') || (c == 'Y'))
  {
   for (;;)
   {
    cursor(15,1); cprintf("Number [%s]:                ",
                           xnam_info[nam-1].number);
    cursor(15,26);
    get_line(buf);
    if (*buf == '\0')
      strcpy(buf,xnam_info[nam-1].number);
    i = 0; 
    p = buf;
    while (*p != '\0')
    {
      if (isdigit(*p))
	ubuf[i++] = *p;
      p++;
    }
    if (i == 10) break;
   }
   s[0] = ubuf[0]; s[1] = ubuf[1]; s[2] = ubuf[2]; s[3] = '\0';
   n = ct_minpack(s);
   xnamblock[(nam-1)*11+2] = n >> 4;
   xnamblock[(nam-1)*11+3] = (n & 0x0f) << 4;
   s[0] = ubuf[3]; s[1] = ubuf[4]; s[2] = ubuf[5]; s[3] = '\0';
   n = ct_minpack(s);
   xnamblock[(nam-1)*11+3] += (n >> 6);
   xnamblock[(nam-1)*11+4] =  (n << 2) & 0xff;
   n = ubuf[6] - '0';
   if (n == 0) n = 10;
   xnamblock[(nam-1)*11+4] += (n >> 2);
   xnamblock[(nam-1)*11+5] = (n & 0x03) << 6;
   s[0] = ubuf[7]; s[1] = ubuf[8]; s[2] = ubuf[9]; s[3] = '\0';
   n = ct_minpack(s);
   xnamblock[(nam-1)*11+5] += (n >> 4);
   xnamblock[(nam-1)*11+6] =  (n & 0x0f) << 4;

    cursor(16,1); cprintf("Sys ID [%s]: ",xnam_info[nam-1].sid);
    get_line(buf);
    if (*buf == '\0')
      strcpy(buf,xnam_info[nam-1].sid);
    n = atoi(buf);
    xnamblock[(nam-1)*11]   = (n >> 8);
    xnamblock[(nam-1)*11+1] = (n & 0xff);

    if (n & 0x01)
      strcpy(ubuf,"0333");
    else
      strcpy(ubuf,"0334");
    cursor(17,1); cprintf("IPCH   [%s]: ",ubuf);
    get_line(buf);
    if (*buf == '\0')
      strcpy(buf,ubuf);
    n = atoi(buf);
    xnamblock[(nam-1)*11+7] = (n >> 8);
    xnamblock[(nam-1)*11+8] = (n & 0xff);

    cursor(18,1); cprintf("ACCOCL [%s]: ",xnam_info[nam-1].olc);
    get_line(buf);
    if (*buf == '\0')
      strcpy(buf,xnam_info[nam-1].olc);
    n = atoi(buf);
    xnamblock[(nam-1)*11+9] = n;

    cursor(19,1); cprintf("GIM    [%s]: ",xnam_info[nam-1].gim);
    get_line(buf);
    if (*buf == '\0')
      strcpy(buf,xnam_info[nam-1].gim);
    n = atoi(buf);
    xnamblock[(nam-1)*11+10] = n;

    gen_naminfo();
    clear_menu_area();
  }
 }
}

/* send xnamblock array to phone.
   generate proper checksum first, and verify and sending. */
save_nam_block()
{
  int i, sum;
  word addr;
  
  /* gen checksum */
  sum = 0x55;
  for (i = 0; i < 65; i++)
    sum = sum + xnamblock[i];
  sum = (~(sum & 0xff)) + 1;
  xnamblock[65] = sum;
  xnamblock[66] = 0x55;

  /* check the checksum! */
  sum = 0;
  for (i = 0; i < 67; i++)
    sum = sum + xnamblock[i];
  sum = sum & 0xff;
  if (sum != 0)
  {
    printf("check of checksum failed!!!");
    exit(1);
  }

  if (ct_model == 900)
  {
    /* enable writes to NAM area */
    addr = 0x7005;
    ct_oki_send_data(54);
    ct_oki_send_data((addr >> 8) & 0xff);
    ct_oki_send_data( addr       & 0xff);
    ct_oki_send_data(1);
  }
  
  /* now save to phone */
  addr = menu_eenamblock;
  for (i = 0; i < 67; i++)
  {
    ct_oki_send_data(54);
    if (ct_model >= 1150)
      ct_oki_send_data(0);
    ct_oki_send_data((addr >> 8) & 0xff);
    ct_oki_send_data( addr       & 0xff);
    ct_oki_send_data(xnamblock[i]);
    ct_get_rss();				/* delay till write is done */
    addr = addr + menu_eenaminc;
    cursor(6,15);  cprintf("%02d",66-i);
  }

  if (ct_model == 900)
  {
    /* disable writes to NAM area */
    addr = 0x7005;
    ct_oki_send_data(54);
    ct_oki_send_data((addr >> 8) & 0xff);
    ct_oki_send_data( addr       & 0xff);
    ct_oki_send_data(0);
  }

  /* reread and test! */
  cursor(6,15);  cprintf("Verifying...");
  get_xnamblock();
  gen_naminfo();
}

void update_nams()
{
  display_menu_title_and_help("Update NAM information\\\
The current NAM information is\\\
being saved in the telephone.");
  cursor(6,1);  cprintf("Please wait...");
  save_nam_block();
  cursor(8,1);  cprintf("Done.  Hit any key to continue.");
  getch();    
}

void edit_memories()
{
  word key;
  byte cmd, data;
  int i, top, mem_ptr, col, sav_pos;
  
  top = 0;
  mem_ptr = 0;
  col = 0;
  display_menu_title_and_help(edit_memories_help);
  edit_display_memories(top);
  edit_select(top,mem_ptr,col,INVERSE_VIDEO);
  for (;;)
  {
#if PC
    if (bioskey(1) != 0)		/* if keypress waiting */
    {
      key = bioskey(0);			/* read keypress codes */
      cmd = key & 0x00ff;		/* get ascii code if any */
      data = (key & 0xff00) >> 8;	/* key function code if any */
#endif
#if UNIX
    if (kbhit())
    {
      cmd = getchar();
      cmd = cmd & 0x7f;
#endif
      if (cmd == ESC)
      {
	edit_select(top,mem_ptr,col,NORMAL_VIDEO);
	break;
      }
      if (cmd == 0)			/* if no ascii code */
      {
	switch (data)			/* convert function code to ascii */
	{
	  case 0x48: cmd = CNP; break;
	  case 0x4b: cmd = CNB; break;
	  case 0x4d: cmd = CNF; break;
	  case 0x50: cmd = CNN; break;
	  case 0x49: cmd = PUP; break;
	  case 0x51: cmd = PDN; break;
	  case 0x47: cmd = '<'; break;
	  case 0x4f: cmd = '>'; break;
	  case 0x52: cmd = INS; break;
	  case 0x53: cmd = DEL; break;
	  deault: break;
	}
      }
      switch (cmd)
      {
	case '<': edit_select(top,mem_ptr,col,NORMAL_VIDEO);
	          mem_ptr = 0; col = 0; top = 0;
	          edit_display_memories(top);
		  edit_select(top,mem_ptr,col,INVERSE_VIDEO);
	          break;
	case '>': edit_select(top,mem_ptr,col,NORMAL_VIDEO);
	          mem_ptr = 0; col = 0; top = max_memories - 20;
	          edit_display_memories(top);
		  edit_select(top,mem_ptr,col,INVERSE_VIDEO);
	          break;
	case TAB: edit_select(top,mem_ptr,col,NORMAL_VIDEO);
		  if (++col > 1) col = 0;
		  edit_select(top,mem_ptr,col,INVERSE_VIDEO);
	          break;
	case CNP: edit_select(top,mem_ptr,col,NORMAL_VIDEO);
	          if (mem_ptr == 0)
	          {
		    if (top > 0) top--;
		    edit_display_memories(top);
		    edit_select(top,mem_ptr,col,INVERSE_VIDEO);
		  }
		  else
		  {
		    mem_ptr--;
		    edit_select(top,mem_ptr,col,INVERSE_VIDEO);
		  }
		  break;
	case CR:  edit_select(top,mem_ptr,col,NORMAL_VIDEO);
		  if (col == 0)
		  {
		    col = 1;
		    edit_select(top,mem_ptr,col,INVERSE_VIDEO);
		    break;
		  }
		  col = 0;
	case CNN: edit_select(top,mem_ptr,col,NORMAL_VIDEO);
                  if (mem_ptr == 19)
	          {
		    if (top < (max_memories - 20)) top++;
		    edit_display_memories(top);
		    edit_select(top,mem_ptr,col,INVERSE_VIDEO);
		  }
		  else
		  {
		    mem_ptr++;
		    edit_select(top,mem_ptr,col,INVERSE_VIDEO);
		  }
		  break;
        case CNF: edit_cursor_forward(&top,&mem_ptr,&col);
		  break;
        case CNB: edit_cursor_backward(&top,&mem_ptr,&col);
		  break;
        case PUP: edit_select(top,mem_ptr,col,NORMAL_VIDEO);
	          top = ((top / 20) * 20) - 20;
		  if (top < 0) top = 0;
		  edit_display_memories(top);
		  mem_ptr = 0;
		  edit_select(top,mem_ptr,col,INVERSE_VIDEO);
		  break;
        case PDN: edit_select(top,mem_ptr,col,NORMAL_VIDEO);
	          top = ((top / 20) * 20) + 20;
		  if (top > (max_memories - 20)) top = max_memories - 20;
		  edit_display_memories(top);
		  mem_ptr = 0;
		  edit_select(top,mem_ptr,col,INVERSE_VIDEO);
		  break;
	case BS:  if (edit_pos == 0) break;
	          edit_pos--;
		  edit_string[edit_pos] = ' ';
	          if (col == 0)
		    if (!isupper(edit_string[0]))
		      edit_string[0] = 'X';
		  if (col == 0) cursor(6+mem_ptr,5+edit_pos);
		    else cursor(6+mem_ptr,16+edit_pos);
		  textattr(INVERSE_VIDEO);
		  putch(edit_string[edit_pos]);
		  textattr(NORMAL_VIDEO);
		  if (col == 0) cursor(6+mem_ptr,5+edit_pos);
		    else cursor(6+mem_ptr,16+edit_pos);
		  edited = TRUE;
	          break;
	case INS: if (col == 0)
		  {
		    for (i = (max_namelen - 2); i >= edit_pos; i--)
		      edit_string[i+1] = edit_string[i];		      
		    edit_string[edit_pos] = ' ';
		    if (!isupper(edit_string[0]))
		      edit_string[0] = 'X';
		  }
		  else
		  {
		    for (i = 30; i >= edit_pos; i--)
		      edit_string[i+1] = edit_string[i];		      
		    edit_string[edit_pos] = ' ';
		  }
	          edited = TRUE;
		  sav_pos = edit_pos;
		  edit_select(top,mem_ptr,col,NORMAL_VIDEO);
		  edit_select(top,mem_ptr,col,INVERSE_VIDEO);
		  edit_pos = sav_pos;
		  if (col == 0)
		    cursor(6+mem_ptr,5+edit_pos);
		  else
		    cursor(6+mem_ptr,16+edit_pos);
	          break;
	case DEL: strcpy(edit_string+edit_pos,edit_string+edit_pos+1);
	          if (col == 0)
		  {
		    if (!isupper(edit_string[0]))
		    {
		      if (islower(edit_string[0]))
			edit_string[0] = toupper(edit_string[0]);
		      else
			edit_string[0] = 'X';
		    }
		    edit_string[max_namelen-1] = ' ';
		    edit_string[max_namelen] = '\0';
		  }
		  else
		  {
		    edit_string[31] = ' ';
		    edit_string[32] = '\0';
		  }
	          edited = TRUE;
		  sav_pos = edit_pos;
		  edit_select(top,mem_ptr,col,NORMAL_VIDEO);
		  edit_select(top,mem_ptr,col,INVERSE_VIDEO);
		  edit_pos = sav_pos;
		  if (col == 0)
		    cursor(6+mem_ptr,5+edit_pos);
		  else
		    cursor(6+mem_ptr,16+edit_pos);
	          break;
	default:  if ((edit_pos == 0) && (cmd == ' '))  /* erase entry */
	          {
		    if (col == 0)
		      strcpy(edit_string,"          ");
		    else
		      strcpy(edit_string,"                                ");
		    edited = TRUE;
		    sav_pos = edit_pos;
		    edit_select(top,mem_ptr,col,NORMAL_VIDEO);
		    edit_select(top,mem_ptr,col,INVERSE_VIDEO);
		    edit_pos = sav_pos;
		    if (col == 0)
		      cursor(6+mem_ptr,5+edit_pos);
		    else
		      cursor(6+mem_ptr,16+edit_pos);
		    break;
		  }	
		  if (col == 0)
		  {
		    if (strchr(LEGAL_ALPHAS,cmd) == NULL) break;
		    if ((edit_pos == 0) && (!isupper(cmd))) break;
		  }
		  if (col == 1)
		  {
	            if (strchr(LEGAL_NUMS,cmd) == NULL) break;
		  }
		  textattr(INVERSE_VIDEO);
		  putch(cmd);
		  textattr(NORMAL_VIDEO);
		  edit_string[edit_pos] = cmd;
		  edit_cursor_forward(&top,&mem_ptr,&col);		      
		  edited = TRUE;
		  break;
      }
    }
  }
}

edit_cursor_forward(top,mem_ptr,col)
  int *top, *mem_ptr, *col;
{
  edit_pos++;
  if (((*col == 0) && (edit_pos > (max_namelen - 1))) ||
      ((*col == 1) && (edit_pos > 31)))
  {
    edit_select(*top,*mem_ptr,*col,NORMAL_VIDEO);
    if (*col == 0) *col = 1;
      else *col = 0;
    edit_select(*top,*mem_ptr,*col,INVERSE_VIDEO);
    return;
  }
  if (*col == 0) cursor(6 + *mem_ptr,5 + edit_pos);
    else if (*col == 1) cursor(6 + *mem_ptr,16 + edit_pos);
}

edit_cursor_backward(top,mem_ptr,col)
  int *top, *mem_ptr, *col;
{
  edit_pos--;
  if (edit_pos < 0)
  {
    edit_select(*top,*mem_ptr,*col,NORMAL_VIDEO);
    if (*col == 0) *col = 1;
      else *col = 0;
    edit_select(*top,*mem_ptr,*col,INVERSE_VIDEO);
    return;
  }
  if (*col == 0) cursor(6 + *mem_ptr,5 + edit_pos);
    else if (*col == 1) cursor(6 + *mem_ptr,16 + edit_pos);
}

edit_select(top,mem_ptr,col,attr)
  int top, mem_ptr, col;
  byte attr;
{
  int end;
  
  if ((attr == NORMAL_VIDEO) && edited)
  {
    /* delete trailing spaces */
    end = strlen(edit_string);
    while (end)
    {
      if ((edit_string[end] == '\0') || (edit_string[end] == ' '))
	edit_string[end] = '\0';
      else
	break;
      end--;
    }
    /* save new string */
    if (col == 0)
      strncpy(memories[top+mem_ptr].name,edit_string,max_namelen);
    else if (col == 1)
      strncpy(memories[top+mem_ptr].number,edit_string,32);
    memories[top+mem_ptr].mod = TRUE;
    memories[top+mem_ptr].got = TRUE;
  }
  if (attr == INVERSE_VIDEO)
  {
    if (col == 0)
      if (max_namelen == 8)
	sprintf(edit_string,"%-8s",memories[top+mem_ptr].name);
      else
	sprintf(edit_string,"%-10s",memories[top+mem_ptr].name);
    else if (col == 1)
      sprintf(edit_string,"%-32s",memories[top+mem_ptr].number);
  }
  edited = FALSE;
  edit_pos = 0;
  if (col == 0) cursor(6+mem_ptr,5);
    else if (col == 1) cursor(6+mem_ptr,16);
  textattr(attr);
  if (col == 0)
  {
    if (max_namelen == 8)
      cprintf("%-8s",memories[top+mem_ptr].name);
    else
      cprintf("%-10s",memories[top+mem_ptr].name);
  }
    else if (col == 1) cprintf("%-32s",memories[top+mem_ptr].number);
  textattr(NORMAL_VIDEO);
  if (col == 0) cursor(6+mem_ptr,5);
    else if (col == 1) cursor(6+mem_ptr,16);
}

edit_display_memories(start)
  int start;
{
  int i, j, row;

  row = 6;
  j = start + 20;
  for (i = start; i < j; i++)
  {
    cursor(row++,1);
    cprintf("%03d %-10s %-32s",i,memories[i].name,memories[i].number);
  }
}

void write_memories_to_file(fname)
  char *fname;
{
  int i, mem, l, ppos;
  FILE *fp;
  char nums[64], *p, *pptr;
  
  if ((fp = fopen(fname,"w")) == NULL) return;
  for (mem = 0; mem < max_memories; mem++)
  {
    if (kbhit())
      if (getch() == ESC) break;
    if (*memories[mem].number == '\0') continue;
    p = memories[mem].number;
    l = strlen(p);
    pptr = strchr(p,'P');
    if (pptr != NULL) ppos = pptr - p; else ppos = -1;
    if (ppos > 0) l = ppos;
    *nums = '\0';
    if ((l == 11) && (*p == '1'))
      sprintf(nums,"1-(%.3s) %.3s-%.4s",p+1,p+4,p+7);
    else if (l == 10)
      sprintf(nums,"(%.3s) %.3s-%.4s",p,p+3,p+6);
    else if (l == 7)
      sprintf(nums,"%.3s-%.4s",p,p+3);
    else if (!strncmp(p,"011",3))
      sprintf(nums,"(%.3s) %s",p,p+3);
    if ((*nums != '\0') && (ppos > 0))
    {
      strcat(nums," ");
      strcat(nums,p+ppos);
    }
    if (*nums == '\0')
      strcpy(nums,p);
    fprintf(fp,"%03d  %-10s  %s\n",mem,memories[mem].name,nums);
  }
  if (!strcmp(fname,"PRN"))
    fprintf(fp,"\014");
  fclose(fp);
}

void print_memories()
{
  cursor(2,1); cputs("Printing memories");
  write_memories_to_file("PRN");
  cursor(2,1); cputs("                    ");
}

int sortcmp(a,b)
  char *a, *b;
{
  if ((*(a+2) == ' ') || (*(a+2) == '\0'))
    return(1);
  if ((*(b+2) == ' ') || (*(b+2) == '\0'))
    return(-1);
  return(strcmp(a+2,b+2));
}

void sort_memories()
{
  int i;
  int end;

  if (ct_model == 900) end = 190;
  if (ct_model >= 1150) end = 90;
  cursor(2,1); cputs("Sorting memories");
  qsort(&memories, 190, MEMORIES_SIZE, sortcmp);
  for (i = 0; i < end; i++)
  {
    memories[i].mod = TRUE;
    memories[i].got = FALSE;
  }
  cursor(2,1); cputs("                      ");
}

void write_memories_file()
{
  char fname[16], fret[16];
  
  strcpy(fname,esn);
  strcat(fname,".MEM");
  get_filename(fname,fret);
  cursor(2,1); cprintf("Writing memories to file %s",fret);
  write_memories_to_file(fret);
  sleep(1);
  cursor(2,1); cputs("                                     ");
}

void read_memories_file()
{
  FILE *fp;
  int mem, len, err;
  char name[16], okname[12], num[64], oknum[40], fname[16], fret[16];
  char *p, *q;
      
  strcpy(fname,esn);
  strcat(fname,".MEM");
  get_filename(fname,fret);
  if ((fp = fopen(fret,"r")) == NULL)
    return;
  cursor(2,1); cputs("Reading memories from file");

  while (fgets(buf,BUFLEN,fp) != NULL)
  {
    mem = atoi(buf);
    p = buf + 4;
    while (isspace(*p)) p++;
    if (*p == '\0') continue;
    strncpy(name,p,10);
    name[max_namelen] = '\0';
    p = p + 10;
    while (isspace(*p)) p++;
    if (*p == '\0') continue;
    num[63] = '\0';
    strset(num,' ');
    strncpy(num,p,64);
    if ((mem < 0) || (mem > (max_memories-1))) continue;
    if (!isupper(*name)) continue;
    p = name;
    q = okname;
    len = 0;
    err = FALSE;
    while (*p != '\0')
    {
      if (strchr(LEGAL_ALPHAS,*p) == NULL)
      {
	p++;
	continue;
      }
      if (len < max_namelen)
	*q++ = *p++;
      len++;
    }
    *q = '\0';
    if (err) continue;

    p = num;
    q = oknum;
    len = 0;
    err = FALSE;
    while (*p != '\0')
    {
      if ((*p == '\n') || (strchr(LEGAL_NUMS,*p) == NULL))
      {
	p++;
	continue;
      }
      if (len < 31)
	*q++ = *p++;
      len++;
    }
    *q = '\0';
    if (err) continue;
    strcpy(memories[mem].name,okname);
    strcpy(memories[mem].number,oknum);
    memories[mem].got = TRUE;
    memories[mem].mod = TRUE;
  }
  cursor(2,1); cputs("                          ");
}

void discharge_battery()
{
  int level, i;
  float volts;
    
  display_menu_title_and_help("Discharge Battery\\\
Hit ESC to stop.\\\
\\\
The phone is transmitting\\\
on an unassigned channel\\\
and the lights are on to\\\
drain the battery quickly.\\\
\\\
The phone will be turned\\\
off when the battery level\\\
becomes very low.");
  cursor(6,1); cputs("Battery Level: ");
  if (ct_model == 900)
  {
  ct_oki_send_data(74);   ct_oki_send_data(00);   /* enter display mode */
  ct_oki_send_data(0xb0); ct_oki_send_data(0x08); /* LEDs on */
  ct_oki_send_data(0x00); ct_oki_send_data(0x3c); /* Display "Battery" */
  ct_oki_send_data(0xff); ct_oki_send_data(0xff); /* exit display mode */
  }
  ct_get_rss();			/* delay till commands are done */
  /* set channel 0, power level high, turn on carrier */
  ct_oki_send_data(9); ct_oki_send_data(0); ct_oki_send_data(0); /* chan 0 */
  ct_oki_send_data(10); ct_oki_send_data(0); /* attenuation 0 */
  ct_oki_send_data(7);			     /* carrier on */
  ct_get_rss();			/* delay till commands are done */
  for (;;)
  {
    if (kbhit())
      if (getch() == ESC) break;
    level = ct_get_battery_level(&volts);
    cursor(6,16);
    cprintf("  %1.2fv  ",volts);
    for (i = 0; i < level; i++)
      putch('+');
    if (level < 1)
    {	
      cursor(7,1); cputs("Battery low.  Power turned off.");
      cleanup();
      exit(0);
      break;
    }
    delay(3000);
  }
  if (ct_model == 900)
  {
  ct_oki_send_data(74);   ct_oki_send_data(00);   /* enter display mode */
  ct_oki_send_data(0xb0); ct_oki_send_data(0x00); /* LEDs off */
  ct_oki_send_data(0x00); ct_oki_send_data(0x00); /* Clear Display */
  ct_oki_send_data(0xff); ct_oki_send_data(0xff); /* exit display mode */
  }
  ct_get_rss();			/* delay till commands done */
  ct_oki_send_data(8);		/* carrier off */
  cursor(6,1); cputs("                    ");
}

debug_print(key)
  byte key;
{
  static count;
  
  fprintf(stdprn,"%02x ",key);
  if (count++ > 15)
  {
    fprintf(stdprn,"\n");
    count = 0;
  }
}

get_filename(msg,buf)
  char *msg, *buf;
{
  int i;
  
  clear_menu_area();
  fflush(stdin);
  cursor(6,1); cprintf("File [%s]: ",msg);
  delay(100);
  while (kbhit()) getch();
  fgets(buf,13,stdin);
  buf[12] = '\0';
  for (i = 0; i < 12; i++)
    if ((buf[i] == '\r') || (buf[i] == '\n') || (buf[i] == ' '))
      buf[i] = '\0';
  if (buf[0] == '\0')
    strcpy(buf,msg);
  clear_menu_area();
}

error(msg)
  char *msg;
{
  cursor(1,1);  cprintf("%-30s",msg);
  cursor(1,strlen(msg)+1);
  getch();
}

status(msg)
  char *msg;
{
  strcpy(status_string,msg);
  cursor(1,1);
  cprintf("%-30s",msg);
}

/* draw or redraw screen */
draw_screen()
{
  int i;
  word chnl;
  
  clrscr();
  cursor(1,49);  cputs("OKI Menu Control Program");
#ifdef NW
  cursor(2,49);  cputs("Copyright 1994 Network Wizards");
#endif
#ifdef CES
  cursor(2,49);  cputs("Cellular Evaluation Systems");
#endif  
#ifdef TSS
  cursor(2,49);  cputs("Tech Support Systems");
#endif  
/*  cursor(4,49);  cputs("Hit ESCAPE to EXIT."); */
#if PC
  cursor(3,1);   cputs("Use \030 and \031 to select menu item.");
  cursor(4,1);   cputs("Hit \032 or ENTER for next menu, \033 for previous.");
#endif
#if UNIX
  cursor(3,1);   cputs("Use ^N and ^P to select menu item.");
  cursor(4,1);   cputs("Hit ^F or CR for next menu, ^B for previous.");
#endif
  cursor(5,1);
  for (i = 1; i < 81; i++)
    putch(HORIZ_LINE);
  for (i = 1; i < 26; i++)
  {
    cursor(i,48); putch(VERT_LINE);
  }
  cursor(5,48); putch(CROSS);
  fflush(stdout);
}

int card_val(val,string)
  int val;
  char *string;
{
/*
  if (val == -1)
  {
    if (cardp) strcpy(string,"On");
      else strcpy(string,"Off");
    return(cardp);
  }
  cardp = val;
  ct_rx_flush();
  for (;;)
  {
    send_key(K_MENU);
    oki_
  }

  oki_menu(
    48 on ,49 off
      
*/
}

void info_disp()
{
  int i, ac, pre, mid, end, sid;
  
  display_menu_title_and_help("Information Display\\\
Hit any key to return\\\
to the Main Menu."); 
  cursor(6,1);  cprintf("Model:          %d",ct_model);
  cursor(7,1);  cprintf("ROM Version:    %s",romver);
  cursor(8,1);  cprintf("ESN (hex):      %s",esn);
  cursor(9,1);  cprintf("  Manuf-Serial: %d-%08ld",esn_man,esn_ser);
  cursor(10,1); cprintf("Power On Msg:   %s",custmsg);
  cursor(11,1); cprintf("Prog Password:  %s",dlrpwd);
  cursor(13,1); cprintf("NAM  Telephone #     System");
  for (i = 0; i < 5; i++)
  {
    cursor(14+i,1);
    cprintf(" %d   %s  %s (%c)",i+1,
            xnam_info[i].number,
	    xnam_info[i].sid,
	    xnam_info[i].sys);
  }

#if UNIX
  fflush(stdout);
#endif
  getch();
  clear_menu_area();
}

void set_custmsg()
{
  int i;
  char c;
  
  for (;;)
  {
    clear_menu_area();
    cursor(2,1); cputs("Set Custom Power On Message");
    cursor(6,1); cprintf("Currently:    %s",custmsg);
    cursor(8,1); cprintf("Do you want to change it? (y/n): ");
    c = getch();
    if ((c == 'y') || (c == 'Y'))
    {
      cursor(9,1);  cprintf("Enter the new message (%d chars max)",
			    max_namelen);
      cursor(10,1); cprintf("12345678");
      if (max_namelen == 10) cprintf("90");
      for (i = 0; i < max_namelen; i++)
	custmsg[i] = '\0';
      cursor(11,1); fgets(custmsg,12,stdin);
      custmsg[max_namelen] = '\0';
      for (i = 0; i < max_namelen; i++)
	if ((custmsg[i] == '\n') || (custmsg[i] == '\0'))
	  custmsg[i] = ' ';
      ct_set_cust_msg(custmsg);
    }
    else 
      break;
  }
  clear_menu_area();
  cursor(2,1); cputs("                           ");
}

get_line(buf)
  char *buf;
{
  char *p;
  
  fgets(buf,80,stdin);
  p = buf;
  while ((*p != '\0') && (*p != '\r') && (*p != '\n')) p++;
  *p = '\0';
}
